unit fROR_RPLoadSave;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, Buttons, uROR_ListView, ActnList,
  fROR_VistARegistries, Menus, uROR_CustomListView, uROR_GridView;

type
  TFormRPLoadSave = class(TForm)
    cmbbName: TComboBox;
    btnOK: TBitBtn;
    btnCancel: TBitBtn;
    lblName: TLabel;
    alMain: TActionList;
    acOpen: TAction;
    acSave: TAction;
    cmbbDirectory: TComboBox;
    lblDirectory: TLabel;
    spbDelete: TSpeedButton;
    acDelete: TAction;
    mnuPopup: TPopupMenu;
    mnuDelete: TMenuItem;
    acRename: TAction;
    mnuRename: TMenuItem;
    lvTemplateNames: TCCRGridView;
    procedure acOpenExecute(Sender: TObject);
    procedure acSaveExecute(Sender: TObject);
    procedure lvTemplateNamesSelectItem(Sender: TObject; Item: TListItem;
      Selected: Boolean);
    procedure acDeleteExecute(Sender: TObject);
    procedure cmbbDirectorySelect(Sender: TObject);
    procedure alMainUpdate(Action: TBasicAction; var Handled: Boolean);
    procedure lvTemplateNamesKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure acRenameExecute(Sender: TObject);
    procedure lvTemplateNamesEdited(Sender: TObject; Item: TListItem;
      var S: String);
    procedure lvTemplateNamesDblClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure lvTemplateNamesClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);

  private
    { fAdmin is True if the user is a registry administrator. }
    fAdmin : Boolean;

    { fInstancePrefix stores the instance name prefix for the report
      parameters templates: 'RegistryIEN:ReportCode:'. }
    fInstancePrefix: String;

    function  getEntity: String;
    function  getInstanceName: String;
    procedure loadParamsList;
    procedure updateNameHistory;

    property InstancePrefix: String read fInstancePrefix;

  public
    function Open(aRegInfo: TRegistryInfo; aPrefix: String): Integer;
    function SaveAs(aRegInfo: TRegistryInfo; aPrefix: String): Integer;

    property Entity: String read getEntity;
    property InstanceName: String read getInstanceName;

  end;

function CheckName(aCaption: String; aGrid: TCCRGridView): TListItem;
function FormRPLoadSave: TFormRPLoadSave;

implementation
{$R *.dfm}

uses
  uROR_Common, uROR_CommonUtils, uROR_Strings, uROR_Utilities, uROR_Broker,
  fROR_Options, uROR_Classes;

const
  dirCommon = 0;
  dirUser   = 1;

var
  fFormRPLoadSave: TFormRPLoadSave = nil;

function CheckName(aCaption: String; aGrid: TCCRGridView): TListItem;
var
  i: integer;
begin
  Result := nil;
  if aGrid.Items.Count > 0 then
  begin
    for i := 0 to aGrid.Items.Count - 1 do
    begin
      if aCaption = aGrid.Items[i].AsString[1] then
      begin
        Result := aGrid.Items[i];
        break;
      end;
    end;
  end;
end;

function FormRPLoadSave: TFormRPLoadSave;
begin
  if not Assigned(fFormRPLoadSave) then
    fFormRPLoadSave := TFormRPLoadSave.Create(Application);
  Result := fFormRPLoadSave;
end;

//////////////////////////////// TFormRPLoadSave \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TFormRPLoadSave.acDeleteExecute(Sender: TObject);
var tmpName: String;
begin
  if Assigned(lvTemplateNames.Selected) then
  begin
    with lvTemplateNames do
    begin
      tmpName := (Selected as TCCRGridItem).AsString[1];
      if (MessageDlg508(rscConfirmation, Format(rscDelete, [tmpName]),
          mtConfirmation, [mbYes,mbNo], 0) = mrYes) then
      begin
        Broker.CreateResults;
        if Broker.CallProc(rpcGUIParameterSet,
          [InstancePrefix+tmpName+U+sReportParamsTemplate, Entity, '@']) then
            Selected.Delete;
      end;
    end;
  end;
end;

procedure TFormRPLoadSave.acOpenExecute(Sender: TObject);
var
  item: TListItem;
begin
  //--- Check if the template with provided name exists
  item := CheckName(cmbbName.Text, lvTemplateNames);
  if not Assigned(item) then
  begin
    //--- Display a warning and cancel the action
    MessageDlg508('', Format(rscNotFound, [cmbbName.Text]), mtWarning, [mbOK], 0);
    ModalResult := mrNone;
  end;
  //--- Update the list of template names used during the session
  if ModalResult <> mrNone then
    updateNameHistory;
end;

procedure TFormRPLoadSave.acRenameExecute(Sender: TObject);
begin
  with lvTemplateNames do
  begin
    if Assigned(Selected) then
    begin
      Selected.Caption := (Selected as TCCRGridItem).AsString[1];
      Selected.EditCaption;
    end;
  end;
end;

procedure TFormRPLoadSave.acSaveExecute(Sender: TObject);
var
  item: TListItem;
begin
  //--- Check if the template with provided name already exists
  item := CheckName(cmbbName.Text, lvTemplateNames);
  if Assigned(item) then
  begin
    //--- Make sure that the names match exactly
    cmbbName.Text := (item as TCCRGridItem).AsString[1];
    //--- Display a warning and request confirmation
    if (MessageDlg508(rscWarning, Format(rscAlreadyExists, [cmbbName.Text]),
          mtWarning, [mbYes,mbNo], 0) <> mrYes) then
      ModalResult := mrNone;
  end;
  //--- Update the list of template names used during the session
  if ModalResult <> mrNone then
    updateNameHistory;
end;

procedure TFormRPLoadSave.alMainUpdate(Action: TBasicAction;
  var Handled: Boolean);
begin
  //--- Disable default behavior of the Ok and Cancel buttons while
  //    editing the caption to avoid undesirable dialog closure
  btnCancel.Cancel := (not lvTemplateNames.IsEditing);
  btnOk.Default    := (not lvTemplateNames.IsEditing);

  //--- Disable actions related to the selected template if there is no selection
  acDelete.Enabled := Assigned(lvTemplateNames.Selected);
  acRename.Enabled := Assigned(lvTemplateNames.Selected);

  //--- Disable the Open/Save button if a name has not been entered/selected
  acOpen.Enabled := (cmbbName.Text <> '');
  acSave.Enabled := (cmbbName.Text <> '');
end;

procedure TFormRPLoadSave.cmbbDirectorySelect(Sender: TObject);
var
  ie: TCCRInterfaceElement;
begin
  with lvTemplateNames do
  begin
    //--- Only registry administrators can edit common templates
    ReadOnly := (not fAdmin) and (cmbbDirectory.ItemIndex = dirCommon);
    if ReadOnly then
    begin
      ie := CCROptions.IE[ieReadOnly];
      Color := ie.Color;
      Font.Color := ie.FontColor;
    end
    else begin
      Color := clWindow;
      Font.Color := clWindowText;
    end;
  end;
  //--- Reload the list of templates
  loadParamsList;
end;

procedure TFormRPLoadSave.FormActivate(Sender: TObject);
begin
  //-- Section 508: Set focused control based upon action to be done
  if CCRScreenReaderActive then
  begin
    if btnOk.Action = acSave then
      cmbbName.SetFocus
    else
      cmbbDirectory.SetFocus;
  end;
end;

procedure TFormRPLoadSave.FormShow(Sender: TObject);
begin
  //-- Section 508: Make fixes to control locations and sizes
  if CCRScreenReaderActive then
  begin
    if (lblName.Top <> (cmbbName.Top + 4)) then
    begin
      lblName.Top := cmbbName.Top + 4;
      cmbbName.Width := cmbbName.Width - (lblName.Width + 5);
      cmbbName.Left := lblName.Left + lblName.Width + 5;
    end;
  end
  else
    cmbbName.SetFocus;
end;

function TFormRPLoadSave.getEntity: String;
begin
  if cmbbDirectory.ItemIndex = dirCommon then
    Result := 'PKG'
  else
    Result := 'USR';
end;

function TFormRPLoadSave.getInstanceName: String;
begin
  //--- Instance name format: {RegistryIEN}:{ReportCode}:{TemplateName}
  Result := InstancePrefix + cmbbName.Text;
end;

procedure TFormRPLoadSave.loadParamsList;
var
  i, n: Integer;
  li: TCCRGridItem;
begin
  Screen.Cursor := crHourGlass;
  try
    Broker.CreateResults;
    //--- Clear the list of template names
    lvTemplateNames.Clear;
    //--- Load the list of template names
    if Broker.CallProc('ROR LIST PARAMETER INSTANCES',
       [sReportParamsTemplate, Entity, InstancePrefix]) then
      begin
        lvTemplateNames.Items.BeginUpdate;
        try
          n := Broker.Results.Count - 1;
          for i := 1 to n do
          begin
              li := lvTemplateNames.Items.Add;
              //-- Section 508: Moved the Dummy Caption field to the front so
              //   JAWS will read the row correctly.
              li.AssignRawData(Broker.Results[i], [-1,1]);
              li.AsString[0] := ' ';
          end;
        finally
          lvTemplateNames.Items.EndUpdate;
        end;
      end;
  finally
    Screen.Cursor := crDefault;
  end;
end;

procedure TFormRPLoadSave.lvTemplateNamesClick(Sender: TObject);
begin
  if Assigned(lvTemplateNames.Selected) and (lvTemplateNames.Selected.Caption = ' ') then
    lvTemplateNames.Selected.Caption := lvTemplateNames.Selected.AsString[1];
end;

procedure TFormRPLoadSave.lvTemplateNamesDblClick(Sender: TObject);
begin
  btnOk.Click;
end;

procedure TFormRPLoadSave.lvTemplateNamesEdited(Sender: TObject;
  Item: TListItem; var S: String);
var
  newItem: TListItem;
  fRen: Boolean;
begin
  if S <> Item.Caption then
  begin
    Broker.CreateResults;
    fRen := True;
    //--- Check if a template with the new name already exists
    newItem := CheckName(S, lvTemplateNames);
    if Assigned(newItem) then
      //--- If it exists but it is the same template (only the
      //    case is different) then ignore it
      if newItem = Item then
        newItem := nil
      //--- If a different template has the same name then
      //    display a warning and request confirmation
      else if MessageDlg508(rscWarning, Format(rscAlreadyExists, [S]),
                mtWarning, [mbYes,mbNo], 0) <> mrYes then
        fRen := False;

    if fRen then
    begin
      //--- Rename the template
      if Broker.CallProc(rpcGUIParameterRename,
           [Entity, sReportParamsTemplate, InstancePrefix+Item.Caption,
           InstancePrefix+S]) then
      begin
        //--- Update the Name feild with the new template name
        cmbbName.Text := S;
        //--- Remove the name of the overwritten template from the list
        if Assigned(newItem) then
          newItem.Delete;
      end
      //--- If the RPC has failed, restore the original template name
      else
        S := Item.Caption;
      (Item as TCCRGridItem).AsString[1] := S;
      (Item as TCCRGridItem).AsString[0] := ' ';
    end;
  end;
end;

procedure TFormRPLoadSave.lvTemplateNamesKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if (not lvTemplateNames.IsEditing) and (lvTemplateNames.SelCount > 0) then
  begin
    if (Key = VK_Delete) and (Shift = []) then
      acDelete.Execute
    else if (Key = VK_F2) then
      acRename.Execute;
  end;
end;

procedure TFormRPLoadSave.lvTemplateNamesSelectItem(Sender: TObject;
  Item: TListItem; Selected: Boolean);
begin
  //--- Update the Name field with the name of selected template
  if Selected then
    cmbbName.Text := (Item as TCCRGridItem).AsString[1];
end;

function TFormRPLoadSave.Open(aRegInfo: TRegistryInfo; aPrefix: String): Integer;
begin
  fAdmin := aRegInfo.Admin;
  fInstancePrefix := aPrefix;

  //--- Customize the dialog box
  Caption := 'Open Report Parameters';
  btnOk.Action := acOpen;
  lblDirectory.Caption := 'Look in:';
  fFormRPLoadSave.HelpContext := 231;

  //--- Load the list of template names and display the dialog box
  loadParamsList;
  Result := ShowModal;
end;

function TFormRPLoadSave.SaveAs(aRegInfo: TRegistryInfo; aPrefix: String): Integer;
begin
  fAdmin := aRegInfo.Admin;
  fInstancePrefix := aPrefix;

  //--- Customize the dialog box
  Caption := 'Save Report Parameters As';
  btnOk.Action := acSave;
  lblDirectory.Caption := 'Save in:';
  fFormRPLoadSave.HelpContext := 230;

  //--- Only registry administrators can modify common templates
  with cmbbDirectory do
  begin
    Enabled := fAdmin;
    if not Enabled then
      ItemIndex := dirUser;
  end;

  //--- Load the list of template names and display the dialog box
  loadParamsList;
  Result := ShowModal;
end;

procedure TFormRPLoadSave.updateNameHistory;
begin
  with cmbbName do
  begin
    //--- Store the current template name
    Items.Insert(0, Text);
    //--- Limit the history size to 15 items
    if Items.Count > 15 then
      Items.Delete(Items.Count);
  end;
end;

end.
